"""
This shows how Traits objects containing Numeric arrays can display views
containing tables and plots based on those arrays, as well as allow the
user to interactively explore the data using the various table and plot
selection tools.

Try selecting regions in a plot by drag selecting an area of the plot and
observing the effect in the table view and the other plots. Similarly, try
selecting regions of the table and observe the effect upon the plots.

Finally, experiment with the various reduction and selection filters at the
top of the table, and observe their effect on the table and plot contents.
"""

# Imports:
from enthought.traits.api \
    import HasPrivateTraits, Array, Tuple, Float
    
from enthought.traits.ui.api \
    import View, HSplit, VSplit, Item, TupleEditor
    
from enthought.model.numeric_ui \
    import Table, Plot
    
from enthought.util.numerix \
    import sin, cos, arange
    
class NumericUIDemo ( HasPrivateTraits ):
    
    # Trait definitions:
    x       = Array
    sin     = Array
    cos     = Array
    xsin    = Array
    x_range = Tuple( 0.0, 12.6, 0.05, 
                     labels = [ 'Start:', 'Stop:', 'Delta:' ],
                     cols   = 3 )
    
    # Traits view definitions:  
    view = View(
               VSplit( 
                   HSplit(
                       Table(), 
                       # Variant: Table( 'x', 'xsin' ) to restrict view to a subset
                       VSplit( 
                           Plot( 'x', 'sin', 'cos', 
                                 dock = 'tab' ),
                           Plot( 'x', 'cos', 'sin', type = 'scatter', 
                                 dock = 'tab' ),
                           Plot( index = 'x', value = 'xsin',
                                 dock = 'tab' )
                           # Variant: Plot( 'x', 'xsin' )
                       ),
                   ),
                   Item( 'x_range', dock = 'vertical', show_label = False ),
                   id = 'splitter'
               ),
               id = 'enthought.traits.ui.demos.Traits UI Demo.Advanced.NumericUI_demo'
           )

    # Initialize the other data once the index data has been defined:           
    def _x_changed ( self, x ):
        self.sin  = sin( x )
        self.cos  = cos( x )
        self.xsin = x * sin( x )
        
    # Handles the user changing the 'start', 'stop' and 'step' values:
    def _x_range_changed ( self, x_range ):
        x0, xn, dx = x_range
        x0, xn     = min( x0, xn ), max( x0, xn )
        
        if (x0 != xn) and (dx != 0.0):
            if ((xn - x0) / dx) > 1000.0:
                dx = (xn - x0) / 1000.0
            self.x = arange( x0, xn, dx )

# Set up the demo:
demo = NumericUIDemo( x_range = ( 0.0, 12.6, 0.05 ) )

if __name__ == "__main__":
    demo.configure_traits()        
    
